function mkcontnt(dashoption,copyoption)
%MKCONTNT - Creates or updates the CONTENTS.M file in the current working directory
%
%Syntax:  mkcontnt(dashoption,copyoption);
%
%Inputs:  dashoption -  0 does not insert a '-' in the H1 lines copied to contents.m (Default)
%                       1 inserts a '-' in the H1 lines copied to contents.m file
%         copyoption -  0 Overwrites contents.m without making a backup copy (Default)
%                       1 Makes a backup copy of "contents.m" to "contents.old".
%
%Output: none  (no variable is output to the calling workspace)
%
%Examples:   mkcontnt     % No backup copy made,  no 'dashes' inserted
%            mkcontnt(1)  % Inserts '-' in the copied H1 lines
%            mkcontnt(1,1)% Inserts '-' in the copied H1 lines, and makes backup copy
%
%Other m-files required: none
%Subfunctions:  putdash
%MAT-files required: none
%
%See also: HELPWIN, DOC


%Full Description (not displayed when typing "help mkcontnt")
%
%MKCONTNT without any arguments copies the H1 line (first comment line) 
%of all m-files found in the current working directory to a file 
%named "contents.m". 
%
%MKCONTNT(1) precedes the descriptive text of the H1 lines within the 
%contents.m file with a dash. With this magic ingredient ('-'), 
%clicking on a highlighted function name within the Helpdesk 
%causes the header help lines for that function to be 
%displayed.  
%
%To enjoy the added functionality brought to the Matlab Helpdesk by 
%the "dashoption", you need to write your H1 lines as follows 
%
%RECOMMENDED H1 LINE FORMAT: 
%
%The first word of the H1 line must be the function name in CAPITAL letters.  
%This is followed by ' - '
%And this is followed by short descriptive text on the same H1 line
%
%EXAMPLES OF H1 LINES from files available on Matlab Central: 
%
% NANCORRCOEF - Compute Pearson's correlation for input vectors that contain NaN values 
% TICKS_FORMAT - Controls the format of tick labels
%
%Take the habit of writing informative H1 lines!!!  
%
%USAGE TIPS: if the user has written several m-files in a directory named
%C:\MATLAB\MY_MFILES, he(she) needs to make this the current working directory,
%using either CD or the path browser, and then either type "mkcontnt",  
%"mkcontnt(1)" or "mkcontnt(1,1)" at the Matlab prompt. Then, assuming
%the MY_MFILES directory in on the matlab path, the user can type 
%"helpwin MY_MFILES" from any directory. The contents.m file for that
%directory will then appear in the ML helpdesk, listing all of the
%m-files of the MY_MFILES directory. If the DASHOPTION = 1 was used, 
%clicking on a function's name then automagically brings on the header 
%help lines for that function.
%
%It is important to note that any fancy editing done to a previous hand-written
%version of contents.m will be lost. Only the top three lines from the
%old version are copied to the new version, but that number can easily
%be increased by minor modifications to the code. You should use the top few
%lines of your contents.m files to describe in general terms what kinds 
%of tasks are performed by the m-files found in the current directory.
%
%Another Helpdesk pointer: The "SEE ALSO" drop down list becomes an  
%active hyperlink for your m-files IF you start the last line of your
%header help lines with the phrase "See also" and IF the function names 
%that follow on the "See also" line are typed in uppercase letters.

%Tested with Matlab 5.2.1,  5.3.1, 6.0, 6.1 for Windows 95, 98 and Windows NT
%
%Author: Denis Gilbert, Ph.D., physical oceanography
%Maurice Lamontagne Institute, Dept. of Fisheries and Oceans Canada
%email: gilbertd@dfo-mpo.gc.ca  Web: http://www.qc.dfo-mpo.gc.ca/iml/
%August 1998;  Last revision: October 05, 2001 
%
%David Harvatin, University of Colorado at Colorado Springs,  
%had the idea of adding dashes to H1 lines and also provided 
%a first draft of the subfunction PUTDASH.
%
%A version of MKCONTNT has been written for the MACINTOSH by 
%J. Arnold Soltz from Draper Laboratory, Cambridge MA, USA, and
%can be obtained directly from him.
%email: soltz@draper.com  

if nargin == 0
    copyoption = 0;  %By default, do not make backup copy of contents.m
    dashoption = 0;  %By default, do not insert ' - ' in H1 lines
elseif nargin == 1
    copyoption = 0;  %By default, do not make backup copy of contents.m
elseif nargin > 2
    error('Too many input arguments')
end  

disp(['Creating contents.m in ' pwd])
%Check if a contents.m file already exists in the current directory
if ~exist([pwd filesep 'contents.m']) % Contents.m does not exist in pwd
    line1 = '%The user should write text describing the m-files found in this directory';
    line2 = '%The top three lines will not be overwritten in subsequent versions of contents.m';
    line3 = '%PLEASE replace these three header lines with MEANINGFUL text';
    %Use the first three lines of your contents.m file to describe in general terms 
    %what kinds of tasks are performed by the m-files found in the current directory
else  
    %Open current version of contents.m and save its first three lines
    fid=fopen('contents.m','rt');
    line1=fgetl(fid);   line2=fgetl(fid); line3=fgetl(fid);
    fclose(fid);
    if copyoption
        %Make a backup copy of contents.m before deleting it.
        copyfile('contents.m','contents.old'); 
    end
    delete contents.m  %Delete current version of contents.m
end

files = what;  % Structure with fields files.m, files.mat, etc.
%Note: the field files.m does not include contents.m (IMPORTANT)
%Do not displace this line of code above or below its present location
%to avoid error messages.

if length(files.m)==0
    %Stop execution if no m-files are present
    warning(['No m-files found in directory ' pwd])
    return  
end

fcontents = fopen('contents.m','wt'); %Write a new version of contents.m
fprintf(fcontents,'%s\n',line1);     %Copy descriptive header text from previous version
fprintf(fcontents,'%s\n',line2);     %Copy descriptive header text (continued)
fprintf(fcontents,'%s\n',line3);     %Copy descriptive header text (continued)
fprintf(fcontents,'%s\n','%  ');%Fourth line is blank for layout purposes

%Make sure all file names are in lowercase to allow proper alphabetical sorting
files.m = sort(lower(files.m));

%Write H1 lines to contents.m if they exist
for i = 1:length(files.m)
    fid=fopen(files.m{i},'rt'); %Cell array of sorted file names
    %Search for first commented line (H1 line)
    count_percent = 0;
    while count_percent < 1 & feof(fid)==0; 
        %True as long as we do not encounter a line with a "%" sign 
        %or reach the end of file
        line = fgetl(fid);
        if length(line) > 0 %Allow for possibility that some lines may be empty
            if ~isempty(findstr(line,'%')) %LOOK for percent sign anywhere in the line
                count_percent = count_percent + 1;
                if sum(~isspace(line)) == 1 % First commented line is empty
                    line = ['%---WARNING--- first commented line of ' upper(files.m{i}) ' is empty'];
                elseif dashoption % Exercises option to insert dash into H1 line
                    line = putdash(line, files.m{i}); % Call subfunction below
                end	
                fprintf(fcontents,'%s\n',line); %Write H1 line to contents.m
            end
        end
        if feof(fid)==1  %End of file encountered without finding a single percent sign
            message_nocommentedline = ['%---WARNING---  ' upper(files.m{i}) ' does not have any commented line'];
            fprintf(fcontents,'%s\n',message_nocommentedline); %Write warning message
        end
    end
    fclose(fid);
end

fclose(fcontents);

%-------------------------------------------------
% SUBFUNCTION PUTDASH
%-------------------------------------------------

function s_out = putdash(s_in, filename)
% PUTDASH  Inserts a dash prior to the descriptive text in the input string.
%
% s_out = putdash(s_in) produces the string s_out with a dash preceding
% the descriptive text of the H1 line represented by the string s_in.
% If the H1 line already contains one occurrence of the form "space-dash-space", 
% then PUTDASH does not insert an additional dash.

s0 = find(~isspace(s_in)); % Indices of nonspaces in s_in
if any(findstr(s_in,' - '))
    %Do not modify the input string s_in if it already contains a ' - '
    s_out = s_in;
elseif s0(2) == 2 % User has letters immediately following "%" symbol in H1 line
    ind = find(diff(s0) ~= 1);
    if isempty(ind)
        %H1 line contains a single word
        s_out = [s_in ' - '];
    else
        ind = s0(ind(1) + 1);  % Index into nonspace that immediately follows 
        %first sequence of one or more spaces
        s_out = [s_in(1:ind-1) '- ' s_in(ind:end)];
    end
else % User has arbitrary # of spaces between "%" symbol and letters in H1 line 
    ind = find(diff(s0) ~= 1);  % Index into nonspace that immediately follows 
    %second sequence of one or more spaces
    if length(ind)==1
        %H1 line contains a single word
        s_out = [s_in ' - '];
    else
        ind = s0(ind(2) + 1); 
        s_out = [s_in(1:ind-1) '- ' s_in(ind:end)];
    end
end
